'use client';

import React, { useState, useEffect } from 'react';
import { useParams } from 'next/navigation';
import { 
  User, 
  Globe, 
  Calendar, 
  FileText, 
  CreditCard, 
  MapPin,
  Clock,
  CheckCircle,
  AlertTriangle
} from 'lucide-react';
import DetailLayout from '@/components/ui/DetailLayout';
import { renderCustomer, renderStatusBadge, visaStatusConfig } from '@/components/ui/TableRenderers';
import type { VisaRequest } from '@/types';
import PermissionGuard from '@/components/guards/PermissionGuard';

interface Comment {
  id: string;
  author: string;
  content: string;
  timestamp: Date;
  type: 'comment' | 'system' | 'status_change';
  isInternal?: boolean;
}

export default function VisaDetailPage() {
  return (
    <PermissionGuard requiredPermissions={['visa_view']}>
      <VisaDetailPageContent />
    </PermissionGuard>
  );
}

function VisaDetailPageContent() {
  const params = useParams();
  const requestId = params.id as string;
  
  const [request, setRequest] = useState<VisaRequest | null>(null);
  const [loading, setLoading] = useState(true);
  const [comments, setComments] = useState<Comment[]>([]);

  useEffect(() => {
    fetchRequestDetails();
    fetchComments();
  }, [requestId]);

  const fetchRequestDetails = async () => {
    try {
      setLoading(true);
      
      // Mock data
      const mockRequest: VisaRequest = {
        id: requestId,
        customerId: 'c1',
        customer: {
          id: 'c1',
          type: 'individual',
          firstName: 'Mehmet',
          lastName: 'Özkan',
          phone: '+905551234567',
          email: 'mehmet@email.com',
          isVip: false,
          isActive: true,
          createdAt: new Date(),
          updatedAt: new Date()
        },
        applicant: {
          id: 'a1',
          firstName: 'Mehmet',
          lastName: 'Özkan',
          passportNumber: 'T12345678',
          passportExpiry: new Date('2028-05-15'),
          nationality: 'Turkish',
          birthDate: new Date('1985-03-20'),
          birthPlace: 'İstanbul',
          phone: '+905551234567',
          email: 'mehmet@email.com',
          occupation: 'Mühendis',
          employer: 'ABC Teknoloji'
        },
        country: 'ABD',
        visaType: 'business',
        applicationDate: new Date('2025-01-10'),
        travelDate: new Date('2025-03-15'),
        returnDate: new Date('2025-03-25'),
        status: 'processing',
        priority: 'high',
        documents: [
          {
            id: 'd1',
            name: 'Pasaport',
            type: 'passport',
            uploadedAt: new Date(),
            fileUrl: '/docs/passport.pdf',
            isRequired: true,
            status: 'approved'
          }
        ],
        requiredDocuments: ['Pasaport', 'Fotoğraf', 'Banka Hesap Özeti', 'Davet Mektubu'],
        appointmentDate: new Date('2025-01-20'),
        appointmentLocation: 'ABD Konsolosluğu İstanbul',
        applicationFee: 160,
        serviceFee: 500,
        totalAmount: 660,
        currency: 'USD',
        agentNotes: 'Acil işlem gerekli - iş seyahati',
        specialRequests: 'Hızlı randevu talep edildi',
        createdAt: new Date('2025-01-10'),
        updatedAt: new Date('2025-01-12')
      };
      
      setRequest(mockRequest);
    } catch (error) {
      console.error('Error fetching request details:', error);
    } finally {
      setLoading(false);
    }
  };

  const fetchComments = async () => {
    const mockComments: Comment[] = [
      {
        id: '1',
        author: 'Sistem',
        content: 'Vize başvurusu oluşturuldu',
        timestamp: new Date('2025-01-10T10:00:00'),
        type: 'system'
      }
    ];
    setComments(mockComments);
  };

  const handleAddComment = (content: string, isInternal: boolean) => {
    const newComment: Comment = {
      id: Date.now().toString(),
      author: 'Agent Mehmet',
      content,
      timestamp: new Date(),
      type: 'comment',
      isInternal
    };
    setComments(prev => [...prev, newComment]);
  };

  const handleStatusChange = (newStatus: string) => {
    if (!request) return;
    setRequest(prev => prev ? { ...prev, status: newStatus as VisaRequest['status'], updatedAt: new Date() } : null);
  };

  const statusOptions = [
    { value: 'pending', label: 'Beklemede' },
    { value: 'document_review', label: 'Belge İncelemesi' },
    { value: 'appointment_scheduled', label: 'Randevu Alındı' },
    { value: 'processing', label: 'İşlemde' },
    { value: 'approved', label: 'Onaylandı' },
    { value: 'rejected', label: 'Reddedildi' },
    { value: 'delivered', label: 'Teslim Edildi' }
  ];

  if (loading || !request) {
    return (
      <div className="min-h-screen bg-gray-50 flex items-center justify-center">
        <div className="animate-pulse space-y-4">
          <div className="h-8 bg-gray-200 rounded w-64"></div>
          <div className="h-64 bg-gray-200 rounded w-96"></div>
        </div>
      </div>
    );
  }

  return (
    <DetailLayout
      title={`Vize Başvurusu #${request.id}`}
      subtitle={`${request.country} - ${request.visaType}`}
      backUrl="/visa/requests"
      recordId={request.id}
      recordType="visa"
      comments={comments}
      onAddComment={handleAddComment}
      onStatusChange={handleStatusChange}
      statusOptions={statusOptions}
      currentStatus={request.status}
    >
      <div className="space-y-6">
        {/* Status & Priority */}
        <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-6">
          <div className="flex items-center justify-between mb-4">
            <h3 className="text-lg font-semibold text-gray-900">Başvuru Durumu</h3>
            <div className="flex space-x-2">
              {renderStatusBadge(request.status, visaStatusConfig)}
              <span className={`px-2 py-1 text-xs font-semibold rounded-full ${
                request.priority === 'high' ? 'bg-red-100 text-red-800' :
                request.priority === 'medium' ? 'bg-yellow-100 text-yellow-800' :
                'bg-green-100 text-green-800'
              }`}>
                {request.priority === 'high' ? 'Yüksek' : 
                 request.priority === 'medium' ? 'Orta' : 'Düşük'} Öncelik
              </span>
            </div>
          </div>
        </div>

        {/* Applicant Info */}
        <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-6">
          <h3 className="text-lg font-semibold text-gray-900 mb-4 flex items-center">
            <User className="h-5 w-5 mr-2" />
            Başvuru Sahibi
          </h3>
          <div className="grid grid-cols-2 gap-6">
            <div>
              <p className="font-medium text-lg">{request.applicant.firstName} {request.applicant.lastName}</p>
              <p className="text-sm text-gray-500">{request.applicant.occupation}</p>
              <p className="text-sm text-gray-500">{request.applicant.employer}</p>
            </div>
            <div>
              <p className="text-sm text-gray-500">Pasaport No:</p>
              <p className="font-mono font-medium">{request.applicant.passportNumber}</p>
              <p className="text-sm text-gray-500">Geçerlilik: {request.applicant.passportExpiry.toLocaleDateString('tr-TR')}</p>
            </div>
          </div>
        </div>

        {/* Visa Details */}
        <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-6">
          <h3 className="text-lg font-semibold text-gray-900 mb-4 flex items-center">
            <Globe className="h-5 w-5 mr-2" />
            Vize Bilgileri
          </h3>
          <div className="grid grid-cols-2 gap-6">
            <div>
              <div className="flex items-center space-x-2 mb-2">
                <MapPin className="h-4 w-4 text-gray-400" />
                <span className="text-sm text-gray-500">Ülke</span>
              </div>
              <p className="text-lg font-semibold">{request.country}</p>
            </div>
            <div>
              <div className="flex items-center space-x-2 mb-2">
                <FileText className="h-4 w-4 text-gray-400" />
                <span className="text-sm text-gray-500">Vize Türü</span>
              </div>
              <p className="font-medium">{request.visaType}</p>
            </div>
          </div>
          
          <div className="mt-6 grid grid-cols-3 gap-4">
            <div>
              <div className="flex items-center space-x-2 mb-2">
                <Calendar className="h-4 w-4 text-gray-400" />
                <span className="text-sm text-gray-500">Başvuru Tarihi</span>
              </div>
              <p className="font-medium">{request.applicationDate.toLocaleDateString('tr-TR')}</p>
            </div>
            <div>
              <div className="flex items-center space-x-2 mb-2">
                <Calendar className="h-4 w-4 text-gray-400" />
                <span className="text-sm text-gray-500">Seyahat Tarihi</span>
              </div>
              <p className="font-medium">{request.travelDate.toLocaleDateString('tr-TR')}</p>
            </div>
            {request.returnDate && (
              <div>
                <div className="flex items-center space-x-2 mb-2">
                  <Calendar className="h-4 w-4 text-gray-400" />
                  <span className="text-sm text-gray-500">Dönüş Tarihi</span>
                </div>
                <p className="font-medium">{request.returnDate.toLocaleDateString('tr-TR')}</p>
              </div>
            )}
          </div>
        </div>

        {/* Appointment */}
        {request.appointmentDate && (
          <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-6">
            <h3 className="text-lg font-semibold text-gray-900 mb-4 flex items-center">
              <Clock className="h-5 w-5 mr-2" />
              Randevu Bilgileri
            </h3>
            <div className="grid grid-cols-2 gap-4">
              <div>
                <span className="text-gray-500">Tarih:</span>
                <p className="font-medium">{request.appointmentDate.toLocaleDateString('tr-TR')}</p>
              </div>
              <div>
                <span className="text-gray-500">Lokasyon:</span>
                <p className="font-medium">{request.appointmentLocation}</p>
              </div>
            </div>
          </div>
        )}

        {/* Documents */}
        <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-6">
          <h3 className="text-lg font-semibold text-gray-900 mb-4 flex items-center">
            <FileText className="h-5 w-5 mr-2" />
            Belgeler ({request.documents.length}/{request.requiredDocuments.length})
          </h3>
          <div className="space-y-3">
            {request.requiredDocuments.map((docName, index) => {
              const uploadedDoc = request.documents.find(d => d.name === docName);
              return (
                <div key={index} className="flex items-center justify-between p-3 bg-gray-50 rounded-lg">
                  <span className="font-medium">{docName}</span>
                  {uploadedDoc ? (
                    <div className="flex items-center space-x-2">
                      <CheckCircle className="h-4 w-4 text-green-600" />
                      <span className="text-sm text-green-600">Yüklendi</span>
                    </div>
                  ) : (
                    <div className="flex items-center space-x-2">
                      <AlertTriangle className="h-4 w-4 text-red-600" />
                      <span className="text-sm text-red-600">Eksik</span>
                    </div>
                  )}
                </div>
              );
            })}
          </div>
        </div>

        {/* Price */}
        <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-6">
          <h3 className="text-lg font-semibold text-gray-900 mb-4 flex items-center">
            <CreditCard className="h-5 w-5 mr-2" />
            Ücret Bilgileri
          </h3>
          <div className="space-y-2">
            <div className="flex justify-between">
              <span>Başvuru Ücreti:</span>
              <span>{request.applicationFee} {request.currency}</span>
            </div>
            <div className="flex justify-between">
              <span>Hizmet Bedeli:</span>
              <span>{request.serviceFee} {request.currency}</span>
            </div>
            <div className="flex justify-between font-bold text-lg border-t pt-2">
              <span>Toplam:</span>
              <span className="text-green-600">{request.totalAmount} {request.currency}</span>
            </div>
          </div>
        </div>

        {/* Visa Result (if approved) */}
        {request.visaNumber && (
          <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-6">
            <h3 className="text-lg font-semibold text-gray-900 mb-4">Vize Bilgileri</h3>
            <div className="grid grid-cols-2 gap-4">
              <div>
                <span className="text-gray-500">Vize Numarası:</span>
                <p className="font-mono font-medium">{request.visaNumber}</p>
              </div>
              {request.validUntil && (
                <div>
                  <span className="text-gray-500">Geçerlilik:</span>
                  <p className="font-medium">{request.validUntil.toLocaleDateString('tr-TR')}</p>
                </div>
              )}
            </div>
          </div>
        )}
      </div>
    </DetailLayout>
  );
}